<?php
/*--------------------------------------------------------------
   GambioWithdrawalsTrend.php 2021-08-19
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;

use Doctrine\DBAL\Connection;
use Gambio\Admin\Modules\Statistics\App\Data\Factory as DataFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory as OverviewFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined\TimespanDropdown;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;
use Gambio\Core\Application\ValueObjects\UserPreferences;

class GambioWithdrawalsTrend extends GambioWidgetDefinition
{
    private const TYPE = 'gambio.withdrawals.trend';
    
    private const VERSION = '0.1.0';
    
    private const ICON = "data:image/png;base64,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";
    
    private const WIDGET_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Widerrufe (Trend)',
        self::LANGUAGE_CODE_ENGLISH => 'Withdrawals Trend',
    ];
    
    private const CATEGORY_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Widerrufe',
        self::LANGUAGE_CODE_ENGLISH => 'Withdrawals',
    ];
    
    /**
     * @var DataFactory
     */
    private $dataFactory;
    
    /**
     * @var UserPreferences
     */
    private $userPreferences;
    
    /**
     * @var Connection
     */
    private $connection;
    
    
    /**
     * @inheritDoc
     */
    public function __construct(
        OverviewFactory $overviewFactory,
        DataFactory     $dataFactory,
        UserPreferences $userPreferences,
        Connection      $connection
    ) {
        $this->dataFactory     = $dataFactory;
        $this->userPreferences = $userPreferences;
        $this->connection      = $connection;
        
        parent::__construct($overviewFactory->createType(self::TYPE),
                            $overviewFactory->createVersion(self::VERSION),
                            $overviewFactory->createIconUsingData(self::ICON),
                            $overviewFactory->useVisualizations()->createAreaChart(),
                            $overviewFactory->useOptions()->createOptions($overviewFactory->useOptions()
                                                                              ->usePredefined()
                                                                              ->createTimespanDropdown($overviewFactory)),
                            $overviewFactory->createTitles($overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_GERMAN),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_GERMAN]),
                                                           $overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_ENGLISH),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_ENGLISH])));
    }
    
    
    /**
     * @inheritDoc
     */
    public function data(Options $options): Data
    {
        $categories    = [];
        $values        = [];
        $categoryTitle = self::CATEGORY_TITLE[$this->userPreferences->languageId()
                                              === self::LANGUAGE_ID_GERMAN ? self::LANGUAGE_CODE_GERMAN : self::LANGUAGE_CODE_ENGLISH];
        
        $timespan    = $this->dataFactory->useTimespan()->createFromTerm($options->getById(TimespanDropdown::ID)
                                                                             ->value());
        $withdrawals = $this->connection->createQueryBuilder()
            ->select([
                         'count(*) AS amount',
                         'UNIX_TIMESTAMP(DATE(withdrawal_date)) AS date',
                     ])
            ->from('withdrawals')
            ->where('withdrawal_date BETWEEN :startDate AND :endDate')
            ->groupBy('date')
            ->orderBy('date')
            ->setParameters([
                                ':startDate' => $timespan->startDate()->format(self::QUERY_TIMESPAN_FORMAT_DATE_START),
                                ':endDate'   => $timespan->endDate()->format(self::QUERY_TIMESPAN_FORMAT_DATE_END),
                            ])
            ->execute()
            ->fetchAll();
        
        foreach ($withdrawals as $withdrawal) {
            $categories[] = $this->dataFactory->useSerialData()->createCategory($withdrawal['date']);
            $values[]     = $this->dataFactory->useSerialData()->createItemValue((int)$withdrawal['amount']);
        }
        
        return $this->dataFactory->useSerialData()->createSerialData($this->dataFactory->useSerialData()
                                                                         ->createCategories(...$categories),
                                                                     $this->dataFactory->useSerialData()
                                                                         ->createItems($this->dataFactory->useSerialData()
                                                                                           ->createItem($this->dataFactory->useSerialData()
                                                                                                            ->createItemTitle($categoryTitle),
                                                                                                        $this->dataFactory->useSerialData()
                                                                                                            ->createItemValues(...
                                                                                                                $values))));
    }
}